//------------------------------------------------------------------------
// Project     : VST SDK
// Version     : 3.6.6
//
// Category    : Interfaces
// Filename    : pluginterfaces/vst/ivstchannelcontextinfo.h
// Created by  : Steinberg, 02/2014
// Description : VST Channel Context Info Interface
//
//-----------------------------------------------------------------------------
// LICENSE
// (c) 2016, Steinberg Media Technologies GmbH, All Rights Reserved
//-----------------------------------------------------------------------------
// This Software Development Kit may not be distributed in parts or its entirety  
// without prior written agreement by Steinberg Media Technologies GmbH. 
// This SDK must not be used to re-engineer or manipulate any technology used  
// in any Steinberg or Third-party application or software module, 
// unless permitted by law.
// Neither the name of the Steinberg Media Technologies nor the names of its
// contributors may be used to endorse or promote products derived from this
// software without specific prior written permission.
// 
// THIS SDK IS PROVIDED BY STEINBERG MEDIA TECHNOLOGIES GMBH "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
// IN NO EVENT SHALL STEINBERG MEDIA TECHNOLOGIES GMBH BE LIABLE FOR ANY DIRECT, 
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
// BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
//----------------------------------------------------------------------------------
#pragma once

#include "pluginterfaces/vst/vsttypes.h"
#include "pluginterfaces/vst/ivstattributes.h"

//------------------------------------------------------------------------
#include "pluginterfaces/base/falignpush.h"
//------------------------------------------------------------------------

//------------------------------------------------------------------------
namespace Steinberg {
namespace Vst {
/** For Channel Context Info Interface */
namespace ChannelContext {

//------------------------------------------------------------------------
//------------------------------------------------------------------------
//------------------------------------------------------------------------

//------------------------------------------------------------------------
//------------------------------------------------------------------------
/** Channel Context Interface. 
\ingroup vstIHost vst365
- [plug imp]
- [extends IEditController]
- [released: 3.6.5]
- [optional]

Allows the host to inform the Plug-in about the context in which the Plug-in is instantiated,
mainly channel based info (color, name, index,...). Index could be defined inside a namespace 
(for example index start from 1 to N for Type Input/Output Channel (Index namespace) and index 
start from 1 to M for Type Audio Channel).\n
As soon as the Plug-in provides this IInfoListener interface, the host will call setChannelContextInfos 
for each change occurring to this channel (new name, new color, new indexation,...)

\section IChannelContextExample Example
\code
tresult PLUGIN_API MyPlugin::setChannelContextInfos (IAttributeList* list)
{
	if (list)
	{
		// optional we can ask for the Channel Name Length
		int64 length;
		if (list->getInt (ChannelContext::kChannelNameLengthKey, length) == kResultTrue)
		{
			...
		}
		
		// get the Channel Name where we, as Plug-in, are instantiated
		String128 name;
		if (list->getString (ChannelContext::kChannelNameKey, name, sizeof (name)) == kResultTrue)
		{
			...
		}

		// get the Channel UID
		if (list->getString (ChannelContext::kChannelUIDKey, name, sizeof (name)) == kResultTrue)
		{
			...
		}
		
		// get Channel Index
		int64 index;
		if (list->getInt (ChannelContext::kChannelIndexKey, index) == kResultTrue)
		{
			...
		}
		
		// get the Channel Color
		int64 color;
		if (list->getInt (ChannelContext::kChannelColorKey, color) == kResultTrue)
		{
			uint32 channelColor = (uint32)color;
			String str;
			str.printf ("%x%x%x%x", ChannelContext::GetAlpha (channelColor),
			ChannelContext::GetRed (channelColor),
			ChannelContext::GetGreen (channelColor),
			ChannelContext::GetBlue (channelColor));
			String128 string128;
			Steinberg::UString (string128, 128).fromAscii (str);
			...
		}

		// get Channel Index Namespace Order of the current used index namespace
		if (list->getInt (ChannelContext::kChannelIndexNamespaceOrderKey, index) == kResultTrue)
		{
			...
		}
	
		// get the channel Index Namespace Length
		if (list->getInt (ChannelContext::kChannelIndexNamespaceLengthKey, length) == kResultTrue)
		{
			...
		}
		
		// get the channel Index Namespace
		String128 namespaceName;
		if (list->getString (ChannelContext::kChannelIndexNamespaceKey, namespaceName, sizeof (namespaceName)) == kResultTrue)
		{
			...
		}

		// get Plugin Channel Location
		int64 location;
		if (list->getInt (ChannelContext::kChannelPluginLocationKey, location) == kResultTrue)
		{
			String128 string128;
			switch (location)
			{
				case ChannelContext::kPreVolumeFader:
					Steinberg::UString (string128, 128).fromAscii ("PreVolFader");
				break;
				case ChannelContext::kPostVolumeFader:
					Steinberg::UString (string128, 128).fromAscii ("PostVolFader");
				break;
				case ChannelContext::kUsedAsPanner:
					Steinberg::UString (string128, 128).fromAscii ("UsedAsPanner");
				break;
				default: Steinberg::UString (string128, 128).fromAscii ("unknown!");
				break;
			}
		}
		
		// do not forget to call addRef () if you want to keep this list
	}
}
\endcode */
//------------------------------------------------------------------------
class IInfoListener: public FUnknown
{
public:
	/** Receive the channel context infos from host. */
	virtual tresult PLUGIN_API setChannelContextInfos (IAttributeList* list) = 0;

	static const FUID iid;
};

DECLARE_CLASS_IID (IInfoListener, 0x0F194781, 0x8D984ADA, 0xBBA0C1EF, 0xC011D8D0)


//------------------------------------------------------------------------
/** Values used for kChannelPluginLocationKey */
//------------------------------------------------------------------------
enum ChannelPluginLocation
{
	kPreVolumeFader = 0,
	kPostVolumeFader,
	kUsedAsPanner
};

//------------------------------------------------------------------------
//------------------------------------------------------------------------
// Colors		
typedef uint32 ColorSpec;	///< ARGB (Alpha-Red-Green-Blue)
typedef uint8 ColorComponent;

inline ColorComponent GetBlue (ColorSpec cs)	{return (ColorComponent)(cs & 0x000000FF); }
inline ColorComponent GetGreen (ColorSpec cs)	{return (ColorComponent)((cs >> 8) & 0x000000FF); }
inline ColorComponent GetRed (ColorSpec cs)		{return (ColorComponent)((cs >> 16) & 0x000000FF); }
inline ColorComponent GetAlpha (ColorSpec cs)	{return (ColorComponent)((cs >> 24) & 0x000000FF); }

//------------------------------------------------------------------------
/** Keys used as AttrID (Attribute ID) in the return IAttributeList of
 * IInfoListener::setChannelContextInfos  */
//------------------------------------------------------------------------
/** string (TChar) [optional]: unique id string used to identify a channel */
const CString kChannelUIDKey = "channel uid";

/** integer (int64) [optional]: number of characters in kChannelUIDKey */
const CString kChannelUIDLengthKey = "channel uid length";

/** string (TChar) [optional]: name of the channel like displayed in the mixer */
const CString kChannelNameKey = "channel name";

/** integer (int64) [optional]: number of characters in kChannelNameKey */
const CString kChannelNameLengthKey = "channel name length";

/** color (ColorSpec) [optional]: used color for the channel in mixer or track */
const CString kChannelColorKey = "channel color";

/** integer (int64) [optional]: index of the channel in a channel index namespace, start with 1 not * 0! */
const CString kChannelIndexKey = "channel index";

/** integer (int64) [optional]: define the order of the current used index namespace, start with 1 not 0!
	For example:
	index namespace is "Input"   -> order 1,
	index namespace is "Channel" -> order 2,
	index namespace is "Output"  -> order 3 */
const CString kChannelIndexNamespaceOrderKey = "channel index namespace order";

/** string (TChar) [optional]: name of the channel index namespace for example "Input", "Output", "Channel", ... */
const CString kChannelIndexNamespaceKey = "channel index namespace";

/** integer (int64) [optional]: number of characters in kChannelIndexNamespaceKey */
const CString kChannelIndexNamespaceLengthKey =	"channel index namespace length";

/** PNG image representation as binary [optional] */
const CString kChannelImageKey = "channel image"; 

/** integer (int64) [optional]: routing position of the Plug-in in the channel (see ChannelPluginLocation) */
const CString kChannelPluginLocationKey = "channel plugin location";
//------------------------------------------------------------------------

} // namespace ChannelContext
} // namespace Vst
} // namespace Steinberg

//------------------------------------------------------------------------
#include "pluginterfaces/base/falignpop.h"
//------------------------------------------------------------------------
